%{
This m-file belongs to the article 'The role of the electrolyte identity on
electrochemical CO2 conversion in the absence of mass transfer limitations'
Authors: Anne Sustronk (1,2) , Nieck E. Benes (1) , Guido Mul (2)
(1) Films in Fluids, Department of Science and Technology, University of 
Twente, Enschede, The Netherlands
(2) Photocatalytic Synthesis Group, Department of Science and Technology, 
University of Twente, Enschede, The Netherlands

The file is used to prepare Figures 1 and 4 of the article.

The next files are required to execute this file: Calculations.m,
ConstantPotential.m, CalH2Data.txt, CalCOData.txt, CalFADataFeb22.txt, and
CalErr.m

Note that the file location in ConstantPotential.m (lines 32, 56, 61 
and 66) should be added before execution of Fig1AndFig4.m.

Brief overview of experimental conditions: Cu hollow fiber working 
electrode, 0.5M electrolyte in MilliQ water, CO2 purged through fiber, 
chronoamperometry

Version date m-file: Feb 22, 2023
%}

clear all, clc, close all

%% Load data files and perform the required calculations

[Data] = Calculations();

NaHCO3 = Data.NaHCO3;
KHCO3 = Data.KHCO3;
CsHCO3 = Data.CsHCO3;
K2SO4 = Data.K2SO4;
KH2PO4 = Data.KH2PO4;

%% Some graph properties

size = 6;                 % Size for plot (width, cm)
SaveFig = 'n';            % Save figures as .emf files y/n
ErrPlot = 'y';            % Plot error bars y/n

%% Compare data NaHCO3, KHCO3, CsHCO3

% Working electrode potential, corrected for internal resistance and the
% corresponding error
xdataCompCat = {NaHCO3.Ereal(:,1),KHCO3.Ereal(:,1),CsHCO3.Ereal(:,1)};
xerrCat = {NaHCO3.Ereal(:,2), KHCO3.Ereal(:,2),CsHCO3.Ereal(:,2)};

% Average of the current density aquired during the last 30 minutes of the 
% experiment and the corresponding error
CDCompCat = {NaHCO3.CDavg30mins(:,1),KHCO3.CDavg30mins(:,1), CsHCO3.CDavg30mins(:,1)};
ynegCat = {NaHCO3.CDavg30mins(:,2),KHCO3.CDavg30mins(:,2),CsHCO3.CDavg30mins(:,2)};
yposCat = {NaHCO3.CDavg30mins(:,2),KHCO3.CDavg30mins(:,2),CsHCO3.CDavg30mins(:,2)};

% Partial current density to CO and the corresponding error
PCDCOCompCat = {NaHCO3.CDCO(:,1),KHCO3.CDCO(:,1),CsHCO3.CDCO(:,1)};
ynegCOCat = {NaHCO3.CDCO(:,2),KHCO3.CDCO(:,2),CsHCO3.CDCO(:,2)};
yposCOCat = {NaHCO3.CDCO(:,2),KHCO3.CDCO(:,2),CsHCO3.CDCO(:,2)};

% Partial current density to H2 and the corresponding error
PCDH2CompCat = {NaHCO3.CDH2(:,1),KHCO3.CDH2(:,1),CsHCO3.CDH2(:,1)};
ynegH2Cat = {NaHCO3.CDH2(:,2),KHCO3.CDH2(:,2),CsHCO3.CDH2(:,2)};
yposH2Cat = {NaHCO3.CDH2(:,2),KHCO3.CDH2(:,2),CsHCO3.CDH2(:,2)};

% Partial current density to FA and the corresponding error
PCDFACompCat = {NaHCO3.CDFA(:,1),KHCO3.CDFA(:,1),CsHCO3.CDFA(:,1)};
ynegFACat = {NaHCO3.CDFA(:,2),KHCO3.CDFA(:,2),CsHCO3.CDFA(:,2)};
yposFACat = ynegFACat;

% Call function to plot cation results
PlotCDComp(xdataCompCat,CDCompCat,xerrCat, ynegCat, yposCat, 101,'CDCompCat',[2 10 size*(4/3) size],{'sb','*k','^r'},{'NaHCO_{3}','KHCO_{3}','CsHCO_{3}'},'SouthEast', SaveFig, ErrPlot)
PlotPCD(xdataCompCat,PCDCOCompCat,xerrCat, ynegCOCat, yposCOCat, 102,'CO PCD CompCat',[10 10 size*(4/3) size],[-1.5 -0.5 -80 5],{'sb','*k','^r'},{'NaHCO_{3}','KHCO_{3}','CsHCO_{3}'},'SouthEast',  -0.65, -4, 'CO', SaveFig, ErrPlot)
PlotPCD(xdataCompCat,PCDH2CompCat,xerrCat, ynegH2Cat, yposH2Cat,103,'H2 PCD CompCat',[10 10 size*(4/3) size],[-1.5 -0.5 -80 5],{'sb','*k','^r'},{'NaHCO_{3}','KHCO_{3}','CsHCO_{3}'},'SouthEast', -0.65, -4, 'H_{2}', SaveFig, ErrPlot)
PlotPCD(xdataCompCat,PCDFACompCat,xerrCat, ynegFACat, yposFACat, 104,'FA PCD CompCat',[10 10 size*(4/3) size],[-1.5 -0.5 -80 5],{'sb','*k','^r'},{'NaHCO_{3}','KHCO_{3}','CsHCO_{3}'},'SouthEast', -0.65, -4, 'FA',SaveFig,ErrPlot)

%% Compare data KHCO3, K2SO4, KH2PO4, (K2CO3)

% Working electrode potential, corrected for internal resistance, and the
% corresponding error
xdataComp = {KHCO3.Ereal(:,1),K2SO4.Ereal(:,1), KH2PO4.Ereal(:,1)};
xerr = {KHCO3.Ereal(:,2), K2SO4.Ereal(:,2),KH2PO4.Ereal(:,2)};

% Average of the current density aquired during the last 30 minutes of the 
% experiment and the corresponding error
CDComp = {KHCO3.CDavg30mins(:,1),K2SO4.CDavg30mins(:,1), KH2PO4.CDavg30mins(:,1)};
yneg = {KHCO3.CDavg30mins(:,2),K2SO4.CDavg30mins(:,2),KH2PO4.CDavg30mins(:,2)};
ypos = {KHCO3.CDavg30mins(:,2),K2SO4.CDavg30mins(:,2),KH2PO4.CDavg30mins(:,2)};

% Partial current density to CO and the corresponding error
PCDCOComp = {KHCO3.CDCO(:,1),K2SO4.CDCO(:,1),KH2PO4.CDCO(:,1)};
ynegCO = {KHCO3.CDCO(:,2),K2SO4.CDCO(:,2),KH2PO4.CDCO(:,2)};
yposCO = {KHCO3.CDCO(:,2),K2SO4.CDCO(:,2),KH2PO4.CDCO(:,2)};

% Partial current density to H2 and the correponding error
PCDH2Comp = {KHCO3.CDH2(:,1),K2SO4.CDH2(:,1),KH2PO4.CDH2(:,1)};
ynegH2 = {KHCO3.CDH2(:,2),K2SO4.CDH2(:,2),KH2PO4.CDH2(:,2)};
yposH2 = {KHCO3.CDH2(:,2),K2SO4.CDH2(:,2),KH2PO4.CDH2(:,2)};

% Partial current density to FA and the corresponding error
PCDFAComp = {KHCO3.CDFA(:,1),K2SO4.CDFA(:,1),KH2PO4.CDFA(:,1)};
ynegFA = {KHCO3.CDFA(:,2),K2SO4.CDFA(:,2),KH2PO4.CDFA(:,2)};
yposFA = ynegFA;

% Call function to plot anion results
PlotCDComp(xdataComp,CDComp, xerr, yneg, ypos, 201,'CD CompAnion',[2 10 size*(4/3) size],{'*k','db','or'},{'KHCO_{3}','K_{2}SO_{4}','KH_{2}PO_{4}'},'SouthEast', SaveFig, ErrPlot)
PlotPCD(xdataComp,PCDCOComp,xerr, ynegCO, yposCO, 202,'CO PCD CompAnion',[10 10 size*(4/3) size],[-1.5 -0.5 -80 5],{'*k','db','or'},{'KHCO_{3}','K_{2}SO_{4}','KH_{2}PO_{4}'},'SouthEast', -0.65, -4, 'CO', SaveFig, ErrPlot)
PlotPCD(xdataComp,PCDH2Comp,xerr, ynegH2, yposH2, 203,'H2 PCD CompAnion',[10 10 size*(4/3) size],[-1.5 -0.5 -80 5],{'*k','db', 'or'},{'KHCO_{3}','K_{2}SO_{4}','KH_{2}PO_{4}'},'SouthEast', -0.65, -4, 'H_{2}', SaveFig, ErrPlot)
PlotPCD(xdataComp,PCDFAComp,xerr, ynegFA, yposFA, 204,'FA PCD CompAnion',[10 10 size*(4/3) size],[-1.5 -0.5 -80 5],{'*k','db', 'or'},{'KHCO_{3}','K_{2}SO_{4}','KH_{2}PO_{4}'},'SouthEast', -0.65, -4, 'FA', SaveFig, ErrPlot)

%% Function files for plotting

%% Plot current density vs potential
function PlotCDComp(xdata,ydata,xerr, yneg,ypos,FigNum,FigName,FigPos,LineSpec,LegendLabel,LegendLoc,SaveFig,ErrPlot)

% Figure properties
fig = figure(FigNum);
fig.Units = 'centimeters';
fig.Color = 'white';
fig.Position = FigPos;
fig.Name = FigName;

% Additional axes properties
ax = gca;
ax.FontName = 'Times New Roman';
ax.TickDir = 'in';
ax.FontSize = 10;
ax.LineWidth = 1;
axis([-1.5 -0.5 -150 5])
ax.XTick = linspace(-1.5,-0.5,6);

% Plot data
hold on, box on
if ErrPlot == 'y'
    cellfun(@errorbar,xdata,ydata,yneg,ypos,xerr, xerr, LineSpec)
else
    cellfun(@plot,xdata,ydata,LineSpec)
end
hold off

% Axis labels and legend properties
xlabel('Potential vs Ag/AgCl')
ylabel('Current density (mA cm^{2}_{geo})')
legend(LegendLabel,'Location',LegendLoc)

% Save figure as .emf yes/no
if SaveFig == 'y'
    saveas(FigNum,FigName,'emf')
else
end

end

%% Plot partial current density vs potential

function PlotPCD(xdata,ydata,xerr, yneg,ypos,FigNum,FigName,FigPos,Axis,LineSpec,LegendLabel,LegendLoc, txtLocX, txtLocY, txtLabel, SaveFig,ErrPlot)

% Figure properties
fig = figure(FigNum);
fig.Units = 'centimeters';
fig.Color = 'white';
fig.Position = FigPos;
fig.Name = FigName;

% Additional axes properties
ax = gca;
ax.FontName = 'Times New Roman';
ax.TickDir = 'in';
ax.FontSize = 10;
ax.LineWidth = 1;
axis(Axis)
ax.XTick = linspace(-1.5,-0.5,6);

% Plot data
hold on, box on
if ErrPlot == 'y'
    cellfun(@errorbar,xdata,ydata,yneg,ypos,xerr, xerr, LineSpec)
else
    cellfun(@plot,xdata,ydata,LineSpec)
end
hold off

% Add labels
t = text(txtLocX, txtLocY ,txtLabel);
t.FontName = 'Times New Roman';
t.FontSize = 10;
t.FontWeight = 'bold';

% Axis lables and legend properties
xlabel('Potential vs Ag/AgCl')
ylabel({'Partial current density';'(mA cm^{2}_{geo})'})
legend(LegendLabel,'Location',LegendLoc)

% Save figure as .emf yes/no
if SaveFig == 'y'
    saveas(FigNum,FigName,'emf')
end

end